#include <algorithm>
#include <list>
#include <stdio.h>

#ifdef MPI_ASM_ALIGN
#include <mpi.h>
#endif /* MPI_ASM_ALIGN */

#include "pairs.h"
#include "exception.h"
#include "nws_algorithm_option.h"

using std::list;
using std::stable_sort;

using align::Sequences;
using align::Pairs;
using core::Exception;
using align::NWSAlgorithmOption;

struct my_pair {
   unsigned int x, y;
   int lenX, lenY;
};

bool comp_idX(my_pair first, my_pair second) {
   return first.x < second.x;
}

bool comp_idY(my_pair first, my_pair second) {
   return first.y < second.y;
}

bool comp_lenX(my_pair first, my_pair second) {
   return first.lenX > second.lenX;
}

bool comp_lenY(my_pair first, my_pair second) {
    return first.lenY > second.lenY;
}

bool comp_all(my_pair first, my_pair second) {
   if (first.lenY != second.lenY)
       return first.lenY > second.lenY;
   else if (first.lenX != second.lenX)
       return first.lenX > second.lenX;
   else if (first.x != second.x)
       return first.x < second.x;
   else
       return first.y < second.y;
}

void Pairs::sort(Sequences *seqs) {
   
   // Only process 0 sorts sequences
   if(NWSAlgorithmOption::mpi_node_id != 0)
        return;
    
   list<my_pair> my_pair_list;
   for (int i = 0; i < pairsCount; i++) {
      my_pair mp;
      mp.x = pair1[i];
      mp.y = pair2[i];
      mp.lenX = seqs->lengths[pair1[i]];
      mp.lenY = seqs->lengths[pair2[i]];
      my_pair_list.push_back(mp);
   }
   
   //my_pair_list.sort(comp_idY);
   //my_pair_list.sort(comp_idX);
   //my_pair_list.sort(comp_lenX);
   //my_pair_list.sort(comp_lenY);
   my_pair_list.sort(comp_all);

   int i = 0;
   list<my_pair>::iterator it;
   for (it = my_pair_list.begin(); it != my_pair_list.end(); ++it) {
      pair1[i] = it->x;
      pair2[i] = it->y;
      i++;
   }
}

Pairs::Pairs(string filename, int bufferSize)
{
    FILE* fd;
    fd = fopen(filename.c_str(), "rb");
    if (fd == NULL)
        throw (new Exception())->setMessage(string("No such file: ") + filename);

    char buffer[bufferSize];
    size_t actualLength;
    pairsCount = 0;
    actualLength = fread(buffer, sizeof(char), bufferSize, fd);
    int carret = 0;
    while ((buffer[carret] >= '0') && (buffer[carret] <= '9') && (carret < actualLength)) 
    {
        pairsCount *= 10;
        pairsCount += buffer[carret++] - '0';
    }
    
    pair1 = new unsigned int[pairsCount];
    pair2 = new unsigned int[pairsCount];
    
    for(unsigned int i=0; i< pairsCount; i++)
    {
        pair1[i] = 0;
        pair2[i] = 0;
    }
    

    int actualPair = -1;
    int first = 1;

    while(actualLength>0)
    {
        for (; carret < actualLength; carret++)
        {
            if (buffer[carret] == '\n')
            {
                actualPair++;
                first = 1;
                continue;
            }
            if ( buffer[carret] == ' ' )
            {
                first = 0;
                continue;
            }
            if((buffer[carret] >= '0') && (buffer[carret] <= '9'))
            {
                if(first)
                {
                    pair1[actualPair] *= 10;
                    pair1[actualPair] += buffer[carret] - '0';
                }
                else
                {
                    pair2[actualPair] *= 10;
                    pair2[actualPair] += buffer[carret] - '0';
                }
            }
        }
        carret = 0;
        actualLength = fread(buffer, sizeof(char), bufferSize, fd);
    }
    
    fclose(fd);
}

Pairs::Pairs()
{
    
}

void Pairs::bcast()
{
    #ifdef MPI_ASM_ALIGN
    HiResTimer timer;
    
//    int y;
//    if(MPI_Bcast(&y, 1, MPI_INT, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
//        throw (new Exception())->setMessage(string("ERROR while handshaking.")); 
    
    MPI_Barrier(MPI_COMM_WORLD);
    
    
        timer.start();
    int tab[10240];
    tab[0] = pairsCount;
    
    if(MPI_Bcast(tab, 10240, MPI_INT, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
        throw (new Exception())->setMessage(string("ERROR while sending pairs.")); 
    
    pairsCount = tab[0];
    
//    if(MPI_Bcast(this, sizeof(Pairs), MPI_CHAR, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
//        throw (new Exception())->setMessage(string("ERROR while sending pairs.")); 
        
    if(NWSAlgorithmOption::mpi_node_id != 0)
    {   
        pair1 = new unsigned int[pairsCount];
        pair2 = new unsigned int[pairsCount];
    }
    
        //SENDING & RECEIVING

        if(MPI_Bcast(pair1, pairsCount, MPI_INT, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
            throw (new Exception())->setMessage(string("ERROR while transfering pairs.")); 

        if(MPI_Bcast(pair2, pairsCount, MPI_INT, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
            throw (new Exception())->setMessage(string("ERROR while transfering pairs.")); 

        timer.stop();
            
    if(NWSAlgorithmOption::mpi_node_id == 0)
        printf("Sending pairs   time=%.2fms, MB=%0.2f\n", timer.getElapsedTime()/1000.0, ((1+pairsCount+pairsCount)*4.0)/(1024.0*1024.0) );
    else
        printf("Receiving pairs time=%.2fms, MB=%0.2f\n", timer.getElapsedTime()/1000.0, ((1+pairsCount+pairsCount)*4.0)/(1024.0*1024.0) );
           
    #endif /* MPI_ASM_ALIGN */
    
}
