/*
 * sequences.cpp
 *
 *  Created on: Oct 8, 2011
 *      Author: michal
 */

#include <math.h>

#ifdef MPI_ASM_ALIGN
#include <mpi.h>
#endif /* MPI_ASM_ALIGN */

#include "sequences.h"
#include "exception.h"
#include "cuda_declarations.h"
#include "algorithm_option.h"
#include "nws_algorithm_option.h"
#include "hi_res_timer.h"

using std::list;

using align::Sequences;
using align::Sequence;
using align::SubstitutionMatrix;
using core::Exception;
using align::NWSAlgorithmOption;

typedef list<Sequence *>::iterator lsit;

Sequences::Sequences(list<Sequence *> *sequences) {
	this->sequences = sequences;
        seqCount = sequences->size();
}


Sequences::~Sequences(){
    if(sequences)
	for(lsit it = sequences->begin(); it != sequences->end(); ++it){
		delete *it;
	}
    delete[] packedSeqs;
}

void Sequences::packSequences(SubstitutionMatrix *sm) {

        // Only process 0 sorts sequences
        if(NWSAlgorithmOption::mpi_node_id != 0)
            return;
    
	this->sm = sm;
	int residuesCount = sm->getResiduesCount();
	int delta = residuesCount - 1;
	this->bits = 1;
	for (int i = 0; i < 8; ++i) {
		if (delta & (1 << i)) {
			this->bits = i + 1;
		}
	}
	mask = 0;
	for (int i = 0; i < bits; ++i) {
		mask <<= 1;
		mask |=  1;
	}

	this->residuesPer32bit = 32 / this->bits;
        
        //if (this->residuesPer32bit == 16)
        //    this->residuesPer32bit = 15;
	

	// INITIALISE "STARTS" ARRAY
	this->starts  = new int[sequences->size()];
	this->lengths = new int[sequences->size()];
	int start = 0;
	lsit it;
	int seqNo;
	longestSeq = 0;
	
	for (it = sequences->begin(), seqNo = 0; it != sequences->end(); ++it, ++seqNo) {
		starts[seqNo] = start;
		lengths[seqNo] = (*it)->length;
		start += (int)ceil((*it)->length*1.0/residuesPer32bit);
		longestSeq = MAX(longestSeq, lengths[seqNo]);
	}
	packedSeqs = new unsigned int[start];
	packedSeqsSize = start * sizeof(unsigned int);

	// PACKING SEQUENCES TO AN ARRAY
	for (it = sequences->begin(), seqNo = 0; it != sequences->end(); ++it, ++seqNo) {
		
		packSequence(&packedSeqs[starts[seqNo]], *it);
//		if(seqNo==268411)
//                {
//                    
//		    printf("%s\n", (*it)->residues);
//		    printf("start CPU %d\n",  starts[268411]);
//		    printf("lengt CPU %d\n", lengths[268411]);
//                }
	}
}

void Sequences::packSequence(unsigned int *memory, Sequence *seqToPack) {

	// IN THIS FUNCTION RESIDUES ARE PACKED IN THE WAY
	// TAHT THE FIRST ONE IS ON THE RIGHT
	//
	// |A|C|G|A|G|C|A|T| -> in this case T is first in the sequence
	//
	// REASON OF SUCH REVERSED ORDER IS TO MAKE
	// UNPACKING ON GPU FASTER

	int currPos = 0;
	int step = residuesPer32bit;
	int iteracja = 0;
	do
	{
		currPos += step;

		if(currPos > seqToPack->length)
		{
			currPos = seqToPack->length;
			step = seqToPack->length % residuesPer32bit;
		}
		memory[iteracja] = 0;
		for(int i=1; i<=step; i++)
		{
			memory[iteracja] <<= bits;
			
			// UPPER CASE if applicable
			if(seqToPack->residues[currPos - i] >= 'a' && seqToPack->residues[currPos - i] <= 'z')
			{
				seqToPack->residues[currPos - i] += - 'a' + 'A';
			}
			
			// CHECKING IF A RESIDUE IS ALLOWED BY SUBSTITUTION MATRIX
			if (sm->convResidueToIdx[ seqToPack->residues[currPos - i] ] == (unsigned int)-1 )
			{
				string message = "ERROR: ";
				message += seqToPack->residues[currPos - i];
				message += " residue not found in SM\n";
				throw (new Exception())->setMessage(message);
			}
			
			memory[iteracja] |= sm->convResidueToIdx[ seqToPack->residues[currPos - i] ];
		}

		iteracja++;
	} while(currPos != seqToPack->length);

}

char* Sequences::unpackSequence(int seqNo)
{
	char *result = new char[lengths[seqNo] + 1];

	unsigned int *memory = &packedSeqs[starts[seqNo]];

	int i;
	for(i=0; i<lengths[seqNo]; i++)
	{
		unsigned int buf = memory[i/residuesPer32bit];
		buf >>= (i%residuesPer32bit) * bits;
		result[i] = sm->convIdxToResidue[ buf & mask ];
	}
	result[i] = 0;

	return result;
}

bool Sequences::testPacking() {
    
        // Only process 0 sorts sequences
        if (NWSAlgorithmOption::mpi_node_id != 0)
            return true;
    
	lsit it;
	int seqNo;
	for (it = sequences->begin(), seqNo = 0; it != sequences->end(); ++it, ++seqNo)
	{
		if( **it != unpackSequence(seqNo) )
                {
                        printf("Seqs packed FAILED\n");
			return false;
                }
	}
        printf("Seqs packed OK\n");
	return true;
}

Sequences::Sequences()
{
    sequences = NULL;
}

void Sequences::bcast(SubstitutionMatrix *sm) 
{
    
    #ifdef MPI_ASM_ALIGN
    HiResTimer timer;
    
//    int y;
//    if(MPI_Bcast(&y, 1, MPI_INT, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
//        throw (new Exception())->setMessage(string("ERROR while handshaking.")); 
    
    MPI_Barrier(MPI_COMM_WORLD);
    
    if(NWSAlgorithmOption::mpi_node_id == 0)
    {        
            //SENDING
            timer.start();
        
            if(MPI_Bcast(this, sizeof(Sequences), MPI_CHAR, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
                throw (new Exception())->setMessage(string("ERROR while sending sequences.")); 
            if(MPI_Bcast(packedSeqs, packedSeqsSize, MPI_CHAR, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
                throw (new Exception())->setMessage(string("ERROR while sending sequences.")); 
            if(MPI_Bcast(starts, seqCount, MPI_INT, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
                throw (new Exception())->setMessage(string("ERROR while sending sequences.")); 
            if(MPI_Bcast(lengths, seqCount, MPI_INT, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
                throw (new Exception())->setMessage(string("ERROR while sending sequences.")); 
            
            timer.stop();
            printf("Sending seqs time=%.2fms, MB=%0.2f\n", timer.getElapsedTime()/1000.0, (sizeof(Sequences)+packedSeqsSize+(seqCount*2)*4.0)/(1024.0*1024.0) );
    }
    else
    {
            //RECEIVING
            timer.start();
        
            if(MPI_Bcast(this, sizeof(Sequences), MPI_CHAR, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
                throw (new Exception())->setMessage(string("ERROR while receiving sequences.")); 
            
            this->sm = sm;
            starts = new int[seqCount];
            lengths = new int[seqCount];
            packedSeqs = new unsigned int[packedSeqsSize / 4];
            
            if(MPI_Bcast(packedSeqs, packedSeqsSize, MPI_CHAR, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
                throw (new Exception())->setMessage(string("ERROR while receiving sequences.")); 
            if(MPI_Bcast(starts, seqCount, MPI_INT, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
                throw (new Exception())->setMessage(string("ERROR while receiving sequences.")); 
            if(MPI_Bcast(lengths, seqCount, MPI_INT, 0, MPI_COMM_WORLD) != MPI_SUCCESS )
                throw (new Exception())->setMessage(string("ERROR while receiving sequences.")); 
            
            timer.stop();
            printf("Receiving seqs time=%.2fms, MB=%0.2f\n", timer.getElapsedTime()/1000.0, (sizeof(Sequences)+packedSeqsSize+(seqCount*2)*4.0)/(1024.0*1024.0) );
    }
    #endif /* MPI_ASM_ALIGN */
}


